#version 120

uniform int time;
uniform float grainIntensity;

varying vec3 vNormal;
varying vec2 vTexCoord;
varying vec3 vPosition;

// Pseudo-random function for noise generation
float random(vec2 st) {
    return fract(sin(dot(st.xy, vec2(12.9898, 78.233))) * 43758.5453123);
}

// Improved noise function for better visual effect
float noise(vec2 st) {
    vec2 i = floor(st);
    vec2 f = fract(st);

    // Four corners in 2D of a tile
    float a = random(i);
    float b = random(i + vec2(1.0, 0.0));
    float c = random(i + vec2(0.0, 1.0));
    float d = random(i + vec2(1.0, 1.0));

    // Cubic Hermite interpolation
    vec2 u = f * f * (3.0 - 2.0 * f);

    return mix(a, b, u.x) +
    (c - a) * u.y * (1.0 - u.x) +
    (d - b) * u.x * u.y;
}

// Fractal Brownian Motion
float fbm(vec2 st) {
    float value = 0.0;
    float amplitude = 0.5;
    float frequency = 3.0;

    // Loop of octaves
    for (int i = 0; i < 5; i++) {
        value += amplitude * noise(st * frequency);
        st *= 2.0;
        amplitude *= 0.5;
    }

    return value;
}

void main() {
    // Use the existing color from the sphere rendering
    vec4 baseColor = gl_Color;

    // Calculate smooth time value
    float timeFloat = float(time) * 0.01;

    // Generate smoke effect using fbm noise
    vec2 smokeCoord = vTexCoord * 3.0 + vec2(timeFloat * 0.2, timeFloat * 0.1);
    float smoke = fbm(smokeCoord);

    // Apply a second layer of noise for more detail
    float detail = fbm(smokeCoord * 2.0 + vec2(timeFloat * -0.15, timeFloat * 0.05));
    smoke = mix(smoke, detail, 0.5);

    // Add some variation based on the position for more interesting smoke patterns
    float posNoise = fbm(vec2(vPosition.x + vPosition.z, vPosition.y + vPosition.x) * 0.1 + timeFloat * 0.05);
    smoke = mix(smoke, posNoise, 0.3);

    // Calculate edge falloff to make edges more transparent
    vec3 viewDir = normalize(-vPosition);
    float edgeFactor = 1.0 - pow(1.0 - abs(dot(normalize(vNormal), viewDir)), 2.0);

    // Create a strong grain effect
    vec2 grainCoord = vTexCoord * 15.0 + timeFloat * 0.3;
    float grain = (random(grainCoord) - 0.5) * grainIntensity;

    // Apply grain to the smoke effect
    float smokeWithGrain = clamp(smoke + grain * 0.3, 0.0, 1.0);

    // Calculate a smoother time-based alpha factor that never goes completely transparent
    // Using sine wave to create a pulsing effect that's always visible
    float pulseAlpha = (sin(timeFloat * 0.7) * 0.3) + 0.6; // Range: 0.3 to 0.9

    // Combine the alpha with noise pattern and edge factor
    float alpha = pulseAlpha * smokeWithGrain * edgeFactor;

    // Ensure alpha never drops below a minimum threshold
    alpha = clamp(alpha, 0.2, 0.9);

    // Apply noise patterns to alpha for varied transparency
    alpha *= mix(0.8, 1.0, fbm(vTexCoord * 2.0 + timeFloat * 0.1));

    // Create the final smoke color by adjusting the base color with the smoke effect
    vec3 finalColor = baseColor.rgb * (0.8 + smokeWithGrain * 0.4);

    // Add visible grain to the final color
    finalColor += vec3(grain * 0.2);

    // Output the final color with alpha
    gl_FragColor = vec4(finalColor, alpha);
}